<?php namespace App\Controllers\Api\Ipn;

use App\Controllers\BaseController;
use App\Libraries\Uid;
use App\Models\AppsModel;
use App\Libraries\Settings;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use App\Models\TransactionsModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesExtraModel;
use App\Models\TransactionsExtraModel;
use App\Models\UsersModel;
use App\Models\SettingsModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class PayPal extends BaseController
{
    protected $paypalUrl = 'https://api-m.paypal.com/v1/notifications/verify-webhook-signature'; 
    protected $method;

    public function __construct()
    {
        $methods = new DepositMethodsModel();
        $this->method = $methods->where([
            "name" => "PayPal",
            "status" => 1
        ])->first();

        if (!$this->method) {
            throw new DatabaseException(lang("Message.message_84"));
        }
    }

    public function index(): ResponseInterface
    {
        try {
            $payload = @file_get_contents('php://input');
            $this->verifyWebhookSignature($payload);

            $event = json_decode($payload, true);

            log_message("debug", "Detect inbox event from PayPal");
            log_message("debug", "Resource detail", $event['resource']);

            
            $customId = null;
            if (isset($event['resource'])) {
                $resource = $event['resource'];
                log_message("debug", "Detect resource array - custom id" . $resource['custom_id'] ?? null);
                $customId = $resource['custom_id'] ?? null;
                $eventType = $event['event_type'] ?? null;
            } elseif (isset($event['id']) && isset($event['purchase_units'])) {
                $resource = $event['purchase_units'][0]['payments']['captures'][0];

                log_message("debug", "Detect purchase_units - custom id" . $resource['custom_id'] ?? null);

                $customId = $resource['custom_id'] ?? null;
                $eventType = $resource['status'] ?? null;
            } else {
                throw new \Exception("Invalid event data");
            }

            if ($customId === null) {
                log_message("error", "Custom ID not found. Custom id event data: ");
                throw new \Exception("Custom ID not found in event data");
            }

            
            if (strpos($customId, 'EXTRA_') === 0) {
                $this->processEventExtra($resource, $eventType);
            } else {
                $this->processEvent($resource, $eventType);
            }

            return $this->respond(["status" => "ok"], 200);
        } catch (\Exception $e) {
            log_message('error', 'Произошла ошибка: pp');
            log_message('error', $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function verifyWebhookSignature($payload)
    {
        $client = service('curlrequest');
        $clientId = $this->method["api_value_1"];
        $clientSecret = $this->method["api_value_2"];
        $auth = base64_encode("$clientId:$clientSecret");

        $response = $client->post($this->paypalUrl, [
            'headers' => [
                'Authorization' => "Basic $auth",
                'Content-Type' => 'application/json'
            ],
            'json' => [
                'transmission_id' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_ID'],
                'transmission_time' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_TIME'],
                'cert_url' => $_SERVER['HTTP_PAYPAL_CERT_URL'],
                'auth_algo' => $_SERVER['HTTP_PAYPAL_AUTH_ALGO'],
                'transmission_sig' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_SIG'],
                'webhook_id' => $this->method["api_value_3"], 
                'webhook_event' => json_decode($payload, true)
            ]
        ]);

        $verification = json_decode($response->getBody(), true);

        log_message("debug", "PayPal verification status" . $verification['verification_status']);

        if ($verification['verification_status'] !== 'SUCCESS') {
            throw new DatabaseException("Invalid webhook signature");
        }
    }

    private function processEvent($resource, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        log_message("debug", "Detect run process event" . $eventType);

        try {
            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                case 'COMPLETED':
                case 'CHECKOUT.ORDER.APPROVED':
                    $this->handlePaymentCaptureCompleted($resource);
                    break;
                default:
                    
                    log_message('info', 'Unhandled PayPal event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }
  
    private function processEventExtra($resource, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        log_message('debug', 'Check event type: '.$eventType);

        try {
            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                case 'COMPLETED':
                case 'CHECKOUT.ORDER.APPROVED':
                    $this->handlePaymentExtraCaptureCompleted($resource);
                    break;
                default:
                    log_message('info', 'Unhandled PayPal extra event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }

    private function handlePaymentCaptureCompleted($resource)
    {
        log_message('debug', 'Start payment completed');
        $subscribes = new SubscribesModel();
        $plans = new PlansModel();
        $transactions = new TransactionsModel();
        $users = new UsersModel();  
        $uid = new Uid();

        $customID = $resource['custom_id'];
        $customValues = explode("_", $customID);

        $plan_id = $customValues[0];
        $user_id = $customValues[1];
        $app_id  = $customValues[2];
        
        log_message('debug', 'Handle custom id transaction:'.$customID);

        
        $plan = $plans->where("id", $plan_id)->select("count,price,build_count")->first();
        if (!$plan) {
            throw new \Exception("Plan not found");
        }
        
        log_message('debug', 'Plan founded'.$plan_id);

        $amount_total = $plan["price"];
        $currentDate = date('Y-m-d');
        $newDate = date('Y-m-d', strtotime('+' . $plan['count'] . ' month', strtotime($currentDate)));
      
        
        $existing_subscribe = $subscribes->where("user_id", $user_id)
                                          ->where("app_id", $app_id)
                                          ->where("expires_at >", strtotime($currentDate))
                                          ->first();

        if ($existing_subscribe) {
            log_message('error', 'Subscribe đã tồn tại cho user_id ' . $user_id . ' và app_id ' . $app_id . ' trong thời hạn');
            return; 
        }
        
        log_message('debug', 'existing_subscribe not found (ok)');

        
        $subscribes->insert([
            "subscribe_external_id" => $resource['id'],
            "customer_external_id"  => $resource['id'],
            "plan_id"               => $plan_id,
            "user_id"               => $user_id,
            "expires_at"            => strtotime($newDate),
            "app_id"                => $app_id,
            "price"                 => $amount_total,
            "uid"                   => $uid->create(),
            "is_active"             => 1,
            "remaining_count"       => $plan["build_count"],
            "method_id"             => $this->method['id']
        ]);
        
        log_message('debug', 'subscribe add to database (ok)');

        
        $transactions->insert([
            'uid' => $uid->create(),
            'amount' => $amount_total,
            'status' => 1,  
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_external_id' => $resource['id'],
            'external_uid' => $resource['id']
        ]);
        
        log_message('debug', 'transaction add to database (ok)');
      
        
    }
  
    private function processRefundCoin($userA_id, $price)
    {
        $users = new UsersModel();
        $settings = new SettingsModel();
        $refund_rate = $settings->where('set_key', 'refund_rate')->select('value')->first()['value'];

        
        $userA = $users->where('id', $userA_id)->select('friend_referral_code')->first();
        if (!$userA || !$userA['friend_referral_code']) {
            log_message('error', 'User A hoặc friend_referral_code không tồn tại.');
            return;
        }

        
        $userB = $users->where('referral_code', $userA['friend_referral_code'])->select('id, refund_coin')->first();
        if (!$userB) {
            log_message('error', 'Không tìm thấy user B tương ứng với friend_referral_code của user A.');
            return;
        }

        
        $refund_coin = round($price * ($refund_rate / 100), 2);

        
        $new_refund_coin = $userB['refund_coin'] + $refund_coin;
        $users->update($userB['id'], ['refund_coin' => $new_refund_coin]);

        log_message('info', 'Refund_coin đã được hoàn thành cho user B (ID: ' . $userB['id'] . ') với số tiền: ' . $refund_coin);
    }
  
    private function handlePaymentExtraCaptureCompleted($resource)
    {
        $projects = new AppsModel();
        $uid = new Uid();

        $plansExtra = new PlansExtraModel();
        $transactions_extra = new TransactionsExtraModel();
        $subscribes_extra = new SubscribesExtraModel();

        $customID = $resource['custom_id'];
        
        if (strpos($customID, 'EXTRA_') === 0) {
            $customID = substr($customID, strlen('EXTRA_'));
        } else {
            throw new \Exception("Invalid custom ID format");
        }
        $customValues = explode("_", $customID);

        
        $plan_extra_id = $customValues[0];
        $user_id = $customValues[1];
        $app_id  = $customValues[2];

        $plan_extra = $plansExtra->where("id", $plan_extra_id)->select("build_count_extra,price")->first();
        if (!$plan_extra) {
            throw new \Exception("Plan extra not found");
        }
        $amount_extra_total = $plan_extra["price"];
        $currentDate = date('Y-m-d');

        
        $subscribes_extra->insert([
            "subscribe_extra_external_id" => $resource['id'],
            "customer_extra_external_id"  => $resource['id'],
            "plan_extra_id"               => $plan_extra_id,
            "user_id"                     => $user_id,
            "app_id"                      => $app_id,
            "price"                       => $amount_extra_total,
            "uid"                         => $uid->create(),
            "is_active"                   => 1,
            "method_id"                   => $this->method['id'],
            "build_count_extra"           => $plan_extra['build_count_extra']
        ]);

        
        $transactions_extra->insert([
            'uid' => $uid->create(),
            'amount' => $amount_extra_total,
            'status' => 1,  
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_extra_external_id' => $resource['id'],
            'external_extra_uid' => $resource['id']
        ]);

        
        $subscribes = new SubscribesModel();
        $subscribes->set('remaining_count', 'remaining_count + ' . $plan_extra["build_count_extra"], false)
                   ->where("user_id", $user_id)
                   ->where("app_id", $app_id)
                   ->update();
    }

    public function capture(): ResponseInterface
    {
        try {
            $clientId = $this->method["api_value_1"];
            $clientSecret = $this->method["api_value_2"];
            $auth = base64_encode("$clientId:$clientSecret");
            $orderId = $this->request->getGet('token');
            $paypalUrl = 'https://api-m.paypal.com/v2/checkout/orders'; 
            $client = service('curlrequest');
            $response = $client->post("$paypalUrl/{$orderId}/capture", [
                'headers' => [
                    'Authorization' => "Basic $auth",
                    'Content-Type' => 'application/json',
                ]
            ]);

            if (!in_array($response->getStatusCode(), [200, 201])) {
                throw new \Exception("Failed to capture PayPal order. Status Code: " . $response->getStatusCode());
            }

            

        } catch (\Exception $e) {
            log_message('error', 'PayPal Capture Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
        $settings = new Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl.'private/profile/subscribe');
    }

    public function captureExtra(): ResponseInterface
    {
        try {
            $clientId = $this->method["api_value_1"];
            $clientSecret = $this->method["api_value_2"];
            $auth = base64_encode("$clientId:$clientSecret");
            $orderId = $this->request->getGet('token');
            $paypalUrl = 'https://api-m.paypal.com/v2/checkout/orders'; 
            $client = service('curlrequest');
            $response = $client->post("$paypalUrl/{$orderId}/capture", [
                'headers' => [
                    'Authorization' => "Basic $auth",
                    'Content-Type' => 'application/json',
                ]
            ]);

            if (!in_array($response->getStatusCode(), [200, 201])) {
                throw new \Exception("Failed to capture PayPal extra order. Status Code: " . $response->getStatusCode());
            }

            

        } catch (\Exception $e) {
            log_message('error', 'PayPal Capture Extra Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
        $settings = new Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl.'private/profile/subscribe_extra');
    }
}
